/**************************************************************************//**
 * @file		hwapi_mod.c
 * @brief		hwapi Kernel Module for QM57 Hardware
 * @date		2013-03-28
 * @copyright		Hatteland Display AS
 ******************************************************************************/

#include <linux/module.h>
#include <linux/fs.h>
#include <linux/version.h>
#include <linux/device.h> 

#include <asm/uaccess.h>
#include <asm/io.h>
#include <asm/delay.h>

#include "hwapi_mod.h"
#define SUCCESS 0
#define TIMEOUT_SMB 2000

#define ERROR_NONE                      0x00
#define ERROR_SMBUS_READ_FAILED         0x03
#define ERROR_SMBUS_BUSY                0x0B

static int Device_Open = 0;
int _Major; 

MODULE_LICENSE("GPL");

static int device_open(struct inode *inode, struct file *file)
{
#ifdef DEBUG
	printk(KERN_INFO "device_open(%p)\n", file);
#endif

	if (Device_Open)
		return -EBUSY;

	Device_Open++;

	try_module_get(THIS_MODULE);
	return SUCCESS;
}

static int device_release(struct inode *inode, struct file *file)
{
#ifdef DEBUG
	printk(KERN_INFO "device_release(%p,%p)\n", inode, file);
#endif

	Device_Open--;

	module_put(THIS_MODULE);
	return SUCCESS;
}

static uint8_t SMBUpdateStatusRegisters(uint16_t BaseAddress)
{
	return inb(BaseAddress);
}

static uint8_t SMBHostBusy(uint16_t SMPort)
{
	uint8_t statReg = SMBUpdateStatusRegisters(SMPort);
	return (statReg & (1 << 0));
}

static uint8_t SMBHostDone(uint16_t SMPort)
{
	uint8_t statReg = SMBUpdateStatusRegisters(SMPort);
	return (statReg & (1 << 7));
}

static uint8_t SMBHostIRQ(uint16_t SMPort)
{
	uint8_t statReg = SMBUpdateStatusRegisters(SMPort);
	return (statReg & (1 << 1));
}

static uint8_t SMBHostDevError(uint16_t SMPort)
{
	uint8_t statReg = SMBUpdateStatusRegisters(SMPort);
	return (statReg & (1 << 2));
}

static uint8_t SMBHostLocked(uint16_t SMPort)
{
	uint8_t statReg = SMBUpdateStatusRegisters(SMPort);
	return (statReg & (1 << 6));
}

/*
static uint8_t SMBCommandOK(uint16_t SMPort)
{
uint8_t statReg = SMBUpdateStatusRegisters(SMPort);

if ((statReg & 0x0002) & (~(statReg & 0x0004)) & (~(statReg & 0x0008)))
{
return 1;
}

return 0;
}
*/

static uint8_t SMBWaitForByteDone(uint16_t BaseAddress)
{
	uint8_t timeout = 0;

	while (!SMBHostDone(BaseAddress) && (timeout++ < TIMEOUT_SMB))
	{
		udelay(10);
	}

	if (timeout >= TIMEOUT_SMB)
		return ERROR_SMBUS_BUSY;

	return ERROR_NONE;
}

static uint8_t SMBWaitForLockRelease(uint16_t BaseAddress)
{
	uint8_t timeout = 0;

	outb(0xFF, BaseAddress);

	while (SMBHostLocked(BaseAddress) && (timeout++ < TIMEOUT_SMB))
	{
		udelay(10);
	}

	if (timeout >= TIMEOUT_SMB)
		return ERROR_SMBUS_BUSY;

	return ERROR_NONE;
}

static uint8_t SMBWaitForBusyRelease(uint16_t BaseAddress)
{
	uint8_t timeout = 0;

	while (SMBHostBusy(BaseAddress) && (timeout++ < TIMEOUT_SMB))
	{
		udelay(10);
	}

	if (timeout >= TIMEOUT_SMB)
	{
		outb(0x02, (BaseAddress + 02));
		return ERROR_SMBUS_BUSY;
	}

	return ERROR_NONE;
}

static uint8_t SMBusBlockRead(uint16_t BaseAddress, uint16_t DeviceID, uint8_t RegIndex, uint8_t *Bytes, uint8_t *nBytes)
{
	int i = 0;
	uint8_t stat;

	if ((stat = SMBWaitForBusyRelease(BaseAddress)) != ERROR_NONE)
		return stat;

	if ((stat = SMBWaitForLockRelease(BaseAddress)) != ERROR_NONE)
		return stat;

	printk(KERN_INFO "Reg %x Base %x \n",RegIndex, BaseAddress);

	outb(RegIndex, (BaseAddress + 0x03));
	outb((uint8_t) DeviceID, (BaseAddress + 0x04));

	inb(BaseAddress + 0x0D);

	outb(0xFE, (BaseAddress + 0x00));    

	{
		outb(0x54, (BaseAddress + 0x02));    
	}

	if ((stat = SMBWaitForByteDone(BaseAddress)) != ERROR_NONE)
		return stat;

	Bytes[0] = inb(BaseAddress + 0x05);
	i++;

	printk(KERN_INFO "i %x bytes %x \n",i, *nBytes);

	while (i < *nBytes)
	{
		if ((stat = SMBWaitForByteDone(BaseAddress)) != ERROR_NONE)
			return stat;

		if (SMBHostDevError(BaseAddress))
			return ERROR_SMBUS_READ_FAILED;

		Bytes[i] = inb(BaseAddress + 0x07);

		if (SMBHostIRQ(BaseAddress))
			break;

		outb(0xC9, (BaseAddress + 0x00));    

		if (SMBHostIRQ(BaseAddress))
			break;

		if (i == ((*nBytes) - 1))
		{
			outb((1 << 5), (BaseAddress + 0x02));    
		}

		i++;
	}

	outb(0xFF, (BaseAddress));    

	if ((stat = SMBWaitForLockRelease(BaseAddress)) != ERROR_NONE)
		return stat;

	return ERROR_NONE;

}

uint8_t message[255];

#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,36)
long device_ioctl(struct file *file,
				  unsigned int ioctl_num,
				  unsigned long ioctl_param)
#else
int device_ioctl(struct inode *inode,
struct file *file,
	unsigned int ioctl_num,
	unsigned long ioctl_param)
#endif
{
	uint8_t *pSysBuf;
 
		if(ioctl_num == IOCTL_READ_SMBUSBLOCK)
		{
			uint16_t BaseAddress;
			uint16_t DeviceID;
			uint8_t RegIndex;
			uint8_t Bytes[32] = {0};
			uint8_t nBytes = 0;
			int i = 0;

			pSysBuf = (uint8_t *) ioctl_param;

			for (i = 0; i < (sizeof (uint16_t) + sizeof (uint16_t) + sizeof (uint8_t) + sizeof (uint8_t)); i++)
				get_user(message[i], pSysBuf + i);

			memcpy(&BaseAddress, (uint8_t *)message, sizeof(uint16_t));
			memcpy(&DeviceID, (uint8_t *)message + sizeof(uint16_t), sizeof(uint16_t));
			memcpy(&RegIndex, (uint8_t *)message + sizeof(uint16_t) + sizeof(uint16_t) , sizeof(uint8_t));
			memcpy(&nBytes, (uint8_t *)message + sizeof(uint16_t) + sizeof(uint16_t) + sizeof(uint8_t), sizeof(uint8_t));

			message[1] = 0;
			message[0] = SMBusBlockRead(BaseAddress, DeviceID, RegIndex, Bytes, &nBytes);

			if (message[0] == ERROR_NONE && nBytes < 32 && nBytes >= 1)
			{
				message[1] = nBytes;
				memcpy(message + 2 * sizeof (uint8_t), Bytes, nBytes * sizeof (uint8_t));

				for (i = 0; i < (sizeof (uint8_t) + sizeof (uint8_t) + nBytes * sizeof (uint8_t)); i++)
					put_user(message[i], pSysBuf + i);

			}
			else
			{
				put_user(message[0], pSysBuf);            
			}

		}

	return SUCCESS;
}

struct file_operations Fops = {
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,36)
	.unlocked_ioctl = device_ioctl,
#else
	.ioctl = device_ioctl,
#endif
	.open = device_open,
	.release = device_release, /* a.k.a. close */
};

static struct class *dev_Class =NULL;

int init_module()
{
	struct device *chr_dev =NULL ;
	
	_Major = register_chrdev(0, DEVICE_FILE_NAME, &Fops);

	if (_Major < 0)
	{
		printk(KERN_ALERT "%s failed with %d\n",
			"Sorry, registering the character device ", _Major);
		return -1;
	}

	dev_Class = class_create(THIS_MODULE,DEVICE_FILE_NAME);

	if( dev_Class == NULL)
	{
		printk( KERN_ALERT "Error!Class couldn't be created!\n" );
		return -1;
	}
	printk( KERN_INFO "Class created!\n" );

	chr_dev = device_create( dev_Class , NULL , MKDEV(_Major,0),NULL,DEVICE_FILE_NAME);

	if( chr_dev == NULL ) 
	{
		printk( KERN_ALERT "Error!Device file couldnt be created\n" );
		return -1;
	}

	printk(KERN_INFO "%s The major device number is %d.\n",
		"Registeration is a success", _Major);

	return 0;
}

void cleanup_module()
{
	int ret ;

	device_destroy(dev_Class,MKDEV(_Major,0));

	class_destroy(dev_Class);       

	unregister_chrdev(_Major, DEVICE_FILE_NAME);
}
